/**
 * This script will add regions to all events on the first selected track,
 * and name those regions using the media file name for each event.
 * The regions will not overlap, even if the events overlap (because overlapping
 * regions will be rejected by DVD Architect's subtitle facility).
 * If a region already exists in the same location, no region is created.
 *
 * If more than one event comes from the same media, this will result in 
 * duplicate region names (which is OK, because you may want the same 
 * subtitle text in more than one place).
 *
 * Use the Edit Details view in Vegas to quickly view, modify, or delete
 * the resulting regions. 
 *
 * You can then use the script that ships with Vegas to convert these
 * region names to a subtitle file that can be imported into DVD Architect
 * (or other DVD authoring program). Thus, the file names of your media
 * can become subtitles, without any work on your part.
 *
 * By John Meyer 4/11/2005
 * (updated 8/1/2005)
 *
 **/

import System;
import System.IO;
import System.Windows.Forms;
import Sony.Vegas;

var evnt : TrackEvent;
var myRegion : Region;
var RegionName : String;


try {

  //Find first selected track
  var track = FindSelectedTrack();
  if (null == track)
      throw "no selected track";

  var eventEnum = new Enumerator(track.Events);
  while (!eventEnum.atEnd()) {
    evnt = TrackEvent(eventEnum.item());
    var MyFilePath = evnt.ActiveTake.MediaPath; 
    var extFileName = Path.GetFileName(MyFilePath);
    var baseFileName = Path.GetFileNameWithoutExtension(extFileName); // Media file name for this event

    var StartTime = evnt.Start;
    var LengthTime = evnt.Length;

    if (evnt.FadeOut.Length.Nanos > 0) {
      LengthTime = LengthTime - evnt.FadeOut.Length;  // Eliminates region overlaps at crossfades.
    }


//    if (evnt.Selected) {  // Add this IF statement, and ending brace, to only add regions for SELECTED events.

    // The RegionExist function checks if a region already exists at this point.
    if (!RegionExist(StartTime.ToMilliseconds(),LengthTime.ToMilliseconds() ) ) {
      myRegion = new Region(StartTime,LengthTime,baseFileName); //Insert a region over this event
      Vegas.Project.Regions.Add(myRegion);
    }

//    } // End if evnt.Selected

    eventEnum.moveNext();
  }  // End while (!eventEnum.atEnd()

} catch (e) {
    MessageBox.Show(e);
}


function FindSelectedTrack() : Track {
  var trackEnum = new Enumerator(Vegas.Project.Tracks);
  while (!trackEnum.atEnd()) {
    var track : Track = Track(trackEnum.item());
    if (track.Selected) {
        return track;
    }
    trackEnum.moveNext();
  }
  return null;
}


function RegionExist(dStart,dLength) : boolean {

  var fmarkerEnum = new Enumerator(Vegas.Project.Regions);

  while (!fmarkerEnum.atEnd()) {
    var fRegion = fmarkerEnum.item();
    var fRegionLength = fRegion.Length.ToMilliseconds();
    var fRegionStart = fRegion.Position.ToMilliseconds();

    if ( (dLength == fRegionLength) && (dStart ==  fRegionStart) ) {
      return 1;
    }
  fmarkerEnum.moveNext();
  }
  return 0;
}